/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/io.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Hardware Dependant Driver");
MODULE_LICENSE("GPL");

static int mn2ws_pcm_hwdep_open(struct snd_hwdep *hw, struct file *file);
static int mn2ws_pcm_hwdep_release(struct snd_hwdep *hw, struct file *file);
static int mn2ws_pcm_hwdep_ioctl(struct snd_hwdep *hw, struct file *file, 
	unsigned int cmd, unsigned long arg);

struct snd_hwdep_ops mn2ws_pcm_hwdep_ops = {
	//.llseek       = , 
	//.read         = , 
	//.write        = , 
	.open         = mn2ws_pcm_hwdep_open, 
	.release      = mn2ws_pcm_hwdep_release, 
	//.poll         = , 
	.ioctl        = mn2ws_pcm_hwdep_ioctl, 
	//.ioctl_compat = , 
	//.mmap         = , 
	//.dsp_status   = , 
	//.dsp_load     = , 
};

int mn2ws_pcm_hwdep_init(struct mn2ws_pcm_dev *d)
{
	if (!d->hwd.desc->enabled) {
		return 0;
	}
	
	spin_lock_init(&d->hwd.spin);
	
	//check the necessary callbacks
	
	//set scaler mode(LPCM x1.0, DEC x0.0)
	d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MAX;
	d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MIN;
	
	return 0;
}

int mn2ws_pcm_hwdep_exit(struct mn2ws_pcm_dev *d)
{
	//do nothing
	return 0;
}

static const char *mn2ws_pcm_hwdep_get_ioctl_name(int cmd)
{
	const char *name = "????";
	
	switch (cmd) {
	case SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_PCM:
		name = "GET_MUTE_PCM";
		break;
	case SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_PCM:
		name = "SET_MUTE_PCM";
		break;
	case SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_DEC:
		name = "GET_MUTE_DEC";
		break;
	case SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_DEC:
		name = "SET_MUTE_DEC";
		break;
	case SNDRV_HWDEP_IOCTL_MN2WS_FORCE_AOUT:
		name = "FORCE_AOUT";
		break;
	default:
		name = "unknown";
		break;
	}
	
	return name;
}

static int mn2ws_pcm_hwdep_open(struct snd_hwdep *hw, struct file *file)
{
	struct mn2ws_pcm_chip *chip = (struct mn2ws_pcm_chip *)hw->private_data;
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	spin_lock(&d->play.spin);
	
	
	spin_unlock(&d->play.spin);
	
	return 0;
}

static int mn2ws_pcm_hwdep_release(struct snd_hwdep *hw, struct file *file)
{
	DPRINTF("%s\n", __func__);
	
	return 0;
}

static int mn2ws_pcm_hwdep_ioctl(struct snd_hwdep *hw, struct file *file, 
	unsigned int cmd, unsigned long arg)
{
	struct mn2ws_pcm_chip *chip = (struct mn2ws_pcm_chip *)hw->private_data;
	struct mn2ws_pcm_dev *d = chip->device;
	int err = -ENOTTY;
	int result;
	
	DPRINTF("%s %d %s\n", __func__, cmd, 
		mn2ws_pcm_hwdep_get_ioctl_name(cmd));
	
	switch (cmd) {
	case SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_PCM:
	{
		int __user *_muted = (void *)arg;
		
		//access check
		if (put_user(0, _muted)) {
			err = -EFAULT;
			goto err_out1;
		}
		
		//return the result
		//PCM scale == 0 -> mute ON, PCM scale != 0 -> mute OFF
		if (d->hwd.scale_pcm == 0) {
			__put_user(1, _muted);
		} else {
			__put_user(0, _muted);
		}
		result = 0;
		
		break;
	}
	case SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_PCM:
	{
		int __user *_muted = (void *)arg;
		int muted;
		
		//access check
		if (get_user(muted, _muted)) {
			err = -EFAULT;
			goto err_out1;
		}
		
		spin_lock(&d->hwd.spin);
		if (muted) {
			d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MIN;
		} else {
			d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MAX;
		}
		
		if (d->hwd.scale_dec != 0) {
			d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MAX;
		}
		
		//ʤ褦 scale ιפ 1.0 ˤ
		if (d->hwd.scale_pcm && d->hwd.scale_dec) {
			d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MAX / 2;
			d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MAX / 2;
		}
		spin_unlock(&d->hwd.spin);
		
		//update HW settings
		spin_lock(&d->play.spin);
		if (d->play.desc->mute) {
			d->play.desc->mute(d);
		}
		spin_unlock(&d->play.spin);
		
		//return the result
		__put_user(muted, _muted);
		result = 0;
		
		break;
	}
	case SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_DEC:
	{
		int __user *_muted = (void *)arg;
		
		//access check
		if (put_user(0, _muted)) {
			err = -EFAULT;
			goto err_out1;
		}
		
		//return the result
		//DEC scale == 0 -> mute ON, DEC scale != 0 -> mute OFF
		if (d->hwd.scale_dec == 0) {
			__put_user(1, _muted);
		} else {
			__put_user(0, _muted);
		}
		result = 0;
		
		break;
	}
	case SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_DEC:
	{
		int __user *_muted = (void *)arg;
		int muted;
		
		//access check
		if (get_user(muted, _muted)) {
			err = -EFAULT;
			goto err_out1;
		}
		
		spin_lock(&d->hwd.spin);
		if (muted) {
			d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MIN;
		} else {
			d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MAX;
		}
		
		if (d->hwd.scale_pcm != 0) {
			d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MAX;
		}
		
		//ʤ褦 scale ιפ 1.0 ˤ
		if (d->hwd.scale_pcm != 0 && d->hwd.scale_dec != 0) {
			d->hwd.scale_pcm = SNDRV_MN2WS_SCALE_MAX / 2;
			d->hwd.scale_dec = SNDRV_MN2WS_SCALE_MAX / 2;
		}
		spin_unlock(&d->hwd.spin);
		
		//update HW settings
		spin_lock(&d->play.spin);
		if (d->play.desc->mute) {
			d->play.desc->mute(d);
		}
		spin_unlock(&d->play.spin);
		
		//return the result
		__put_user(muted, _muted);
		result = 0;
		
		break;
	}
	case SNDRV_HWDEP_IOCTL_MN2WS_FORCE_AOUT:
		//---- Only for debug ----
		//Overwrite the settings of AOUT mode
		if (d->hwd.desc->force_aout) {
			d->hwd.desc->force_aout(d);
		}
		
		result = 0;
		
		break;
	default:
		//default
		PRINTF_WARN("ch%d_h: unknown ioctl %d(%s).\n", 
			d->ch, cmd, mn2ws_pcm_hwdep_get_ioctl_name(cmd));
		result = -ENOTTY;
		break;
	}
	
	DPRINTF("scale_pcm:%d, scale_dec:%d\n", 
		d->hwd.scale_pcm, d->hwd.scale_dec);
	
	return result;
	
err_out1:
	return err;
}

